unit Mathimge;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics,
  Controls, Forms, Dialogs, ExtCtrls;

type
  colorarraytype=array[0..17] of longint;

  PSurface=^TSurface;

  PD3Point=^TD3Point;

  TD3point=record
             xw,yw:integer;
             dist:single;
             doshow:boolean;
            end;

  PPointarray=^TPointarray;

  TPointarray=array[0..200] of PD3Point;

  TSurface=array[0..200] of PPointarray;

  TMathImage = class(timage)
  private
    x1d2,x2d2,y1d2,y2d2,ax,bx,ay,by:extended;
    axisglb:boolean;
    x1d3,x2d3,y1d3,y2d3,z1d3,z2d3,alpha:extended;
    zrd3,yrd3:extended;
    xm,ym:integer;
    basex,basey,basez,frontx,fronty,frontz,vd:extended;
    arad,tana,thetaz,thetay,sinz,siny,cosz,cosy,
    ad3,bxd3,byd3,bzd3,ap,bxp,byp:extended;
    rightz, righty:extended;
    surface:psurface;
    surfaceallocated:boolean;
    maxth,maxtw:integer;
    fversion:string;
    procedure setversion(value:string);
    procedure resetworld;
    procedure d3resetworld;
    procedure setaxisglb(a:boolean);
    procedure setx1d2(x:extended);
    procedure setx2d2(x:extended);
    procedure sety1d2(x:extended);
    procedure sety2d2(x:extended);
    procedure setx1d3(x:extended);
    procedure sety1d3(x:extended);
    procedure setx2d3(x:extended);
    procedure sety2d3(x:extended);
    procedure setz1d3(x:extended);
    procedure setz2d3(x:extended);
    procedure setvd(x:extended);
    procedure setzrd3(x:extended);
    procedure setyrd3(x:extended);
    procedure setalpha(x:extended);
    function scalar(xb,yb,zb:extended):extended;
    procedure blockx(x:extended;var xb:extended);
    procedure blocky(y:extended;var yb:extended);
    procedure blockz(z:extended;var zb:extended);
    procedure project(xb,yb,zb:extended; var u,v:extended);
    procedure makeradians;
    function dist(xb,yb,zb:extended):extended;
    procedure findbase(var i1,i2,i3:integer);
    procedure initworld;
    procedure drawoneaxis(x1,y1,z1,x2,y2,z2:extended;c:string);
    function inspanned(p1,p2,p3,q:td3point):boolean;
    { Private declarations, never mind }
  protected
    { Protected declarations }
{---------------------*********************************--------------------------}
{                               THE IMPORTANT STUFF                                                 }
{---------------------*********************************--------------------------}
  public
    property D2WorldX1: extended read x1d2 write setx1d2;
    property D2WorldX2: extended read x2d2 write setx2d2;
    property D2WorldY1: extended read y1d2 write sety1d2;
    property D2WorldY2: extended read y2d2 write sety2d2;
    {The above set the boundary for the 2-d-drawing world}

    property D2Axes:boolean read axisglb write setaxisglb;
    {If true, space is reserved on the image canvas to include axes.
    You need to call drawaxes in order to actually draw any.}

    property D3WorldX1: extended read x1d3 write setx1d3;
    property D3WorldX2: extended read x2d3 write setx2d3;
    property D3WorldY1: extended read y1d3 write sety1d3;
    property D3WorldY2: extended read y2d3 write sety2d3;
    property D3WorldZ1: extended read z1d3 write setz1d3;
    property D3WorldZ2: extended read z2d3 write setz2d3;
    {These set the boundaries for the 3-d-drawing world. When graphed,
     the world box is normalized so its longest side has length 2, and
     the other sides have lengthes according to the true aspect ratio of
     the bounds you specify. The box is then projected onto the image
     according to the settings of D3ViewDist, D3ViewAngle, D3Zrotation,
     D3yrotation}

    property D3Zrotation: extended read zrd3 write setzrd3;
    {Angle of viewpoint with the x-axis (how much it's rotated
     about the z-axis)}

    property D3Yrotation: extended read yrd3 write setyrd3;
    {Angle of viewpoint with the z-axis (how much the viewpoint is
     rotated about the y-axis)}

    property D3ViewDist: extended read vd write setvd;
    {Distance of viewpoint to the center of the d3-world, which has been
     normalized so its longest side has length 2. }

    property D3ViewAngle:extended read alpha write setalpha;
    {Opening angle of the lens of the viewpoint. Large D3ViewAngle combined with
     small D3ViewDist give fish eye effect. See project SurfDemo to get a feel.}

    constructor create(AOwner:TComponent); override;
    destructor destroy; override;
     {The destructor also disposes of any allocated surface memory}

    {The following are the methods for 2-d graphing:}
    procedure setworld(x1,y1,x2,y2:extended);
    {set world range in one step}

    procedure reset;
    {Call whenever the image is being resized. It adjusts the world to pixel
    scaling and resizes the bitmap}

    procedure setcolor(color:longint);
    {Short for canvas.pen.color:=color}

    function getcolor:longint;
    {Short for result:=canvas.pen.color;}

    function WindowX(X : extended):integer;
    {Translates World x to pixel x}

    function WindowY(Y : extended):integer;
    {Translates World y to pixel y}

    function worldx(xs:integer):extended;
    function worldy(ys:integer):extended;
    {Translate pixel to world}

    function norm(x,y:extended):extended;
    {Length of vector (x,y)}

    procedure clear;
    {Erases current picture, puts a new bitmap into picture}

    procedure DrawPoint(X, Y : extended);
   {puts a pixel with world coordinates (x,y) on the screen. Color
    is the currently selected pen color -> setcolor}

    procedure MovetoPoint(X,Y:extended);
   {moves the cursor to the point with world coordinates (x,y)}

    procedure DrawLine(X1, Y1, X2, Y2 : extended);
   {Draws a line from (x1,y1) to (x2,y2) in world coordinates}

    procedure DrawLineto(X,Y:extended);
   {draws a line from the current cursor position (see MovetoPoint) to
   point (x,y) in world coordinates}

    procedure DrawAxes(xlabel,ylabel:string;
                        zerolines:boolean;
                        axescolor,zerolinescolor:longint);
    {Draws Axes at the left and bottom boundary of the image. Ticks and
     labelling of numeric values are done automatically. xlabel, ylabel is
     text that goes to the end of the axes. Zerolines=true draws lines x=0,
     y=0. Axescolor,ZelolinesColor are selfexplaining.}

    procedure DrawVector(x,y,a,b:extended);
    {Draws a vector (a,b) at base (x,y)}

   {D3Graphics procedures:}
    procedure d3setworld(x1,y1,z1,x2,y2,z2,vdist
          ,vangle:extended;zrot,yrot:extended);
    {Sets all d3-graphic parameters in one step}

    procedure d3window(x,y,z:extended; var xs,ys:integer);
    {translates world-(x,y,z) to pixel-(xs,ys)}

    procedure d3moveto(x,y,z:extended);
    procedure d3drawpoint(x,y,z:extended);
    procedure d3drawline(x1,y1,z1,x2,y2,z2:extended);
    procedure d3drawlineto(x,y,z:extended);
    {The 3-d-analogues of the line procedures}

    procedure d3drawaxes(c1,c2,c3:string);
    {Draws axes (without ticks yet) at the bondary of the world box
    and puts c1,c2,c3 on their ends}

    procedure d3drawworldbox;
    {Draws the box the current world resides in, with the 3 sides facing
    the viewer left open}

    procedure d3drawzerocross;
    {Draws lines x=y=0, x=z=0, y=z=0}

    function d3distancetoviewer(x,y,z:extended):extended;
    {The (scaled) distance of the viewpoint to point (x,y,z). Can be
    used to see whether something is visible}

    {Surface Routines:}
    procedure d3CreateSurfaceMem(xmesh,ymesh:integer;var created:boolean);
    {Allocates Memory for surface drawing. xmesh, ymesh are the number of
    grid points the surface has. Needs to be called before any other
    surface routines. At the next call to d3CreateSurfaceMem, the current one
    gets disposed of. The maximal value for xmesh or ymesh is 200, more than
    you'd need, usually.}

    procedure d3MakeSurfacePoint(i,j:integer;x,y,z:extended;var made:boolean);
    {Assign the world point (x,y,z) to the surface at grid (i,j)}

    procedure d3MakeHideInfo(deaf:boolean);
    {Computes visibility of points on the current surface under current
     viewpoint settings. Call before drawing a surface filled. Though the
     algorithm is pretty cheap, it still takes long. If you want to enable
     multitasking (application.processmessages) while it computes, set deaf
     to false.}

    function d3visible(i,j:integer):boolean;
    {Tells whether the surface point at grid (i,j) is visible. (Only
    meaningful after d3MakeHideInfo has been done)}

    procedure d3DrawSurface(fill,deaf:boolean);
    {Draw the current surface. Fill=false gives a wire frame (fast)
    Fill=true displays it filled with the current brush color.
    If MakeHideInfo has been called, non visible parts will appear
    (more or less) hidden. Deaf=true stops message processing while it draws.}

    { Public declarations }
  published
  property version:string read fversion write setversion;
  {Fake property to display the version of the component}
    { Published declarations }
  end;

const colorarray:colorarraytype = (clblack,clnavy,clmaroon,
 clolive,clpurple,clteal,clgray,clgreen,clred,clblue,clfuchsia,
 cllime,claqua,clyellow,clsilver,16777088,8454016,clwhite);
 {A handy array to circle through the system colors}

procedure Register;


implementation

procedure Register;
begin
  RegisterComponents('Samples', [TMathImage]);
end;

var hregion:HRgn;

procedure TMathImage.setversion;
begin
  fversion:='1.1  Oct 96';
end;

procedure TMathImage.setx1d2;
begin
  x1d2:=x; resetworld;
end;
procedure TMathImage.setx2d2;
begin
  x2d2:=x; resetworld;
end;
procedure TMathImage.sety1d2;
begin
  y1d2:=x; resetworld;
end;
procedure TMathImage.sety2d2;
begin
  y2d2:=x; resetworld;
end;
procedure TMathImage.setx1d3;
begin
  x1d3:=x; d3resetworld;
end;
procedure TMathImage.setx2d3;
begin
  x2d3:=x; d3resetworld;
end;
procedure TMathImage.sety1d3;
begin
  y1d3:=x; d3resetworld;
end;
procedure TMathImage.sety2d3;
begin
  y2d3:=x; d3resetworld;
end;
procedure TMathImage.setz1d3;
begin
  z1d3:=x; d3resetworld;
end;
procedure TMathImage.setz2d3;
begin
  z2d3:=x; d3resetworld;
end;
procedure TMathImage.setvd;
begin
  vd:=x; d3resetworld;
end;
procedure TMathImage.setalpha;
begin
  alpha:=x; d3resetworld;
end;
procedure TMathImage.setzrd3;
begin
  zrd3:=x; d3resetworld;
end;
procedure TMathImage.setyrd3;
begin
  yrd3:=x; d3resetworld;
end;

Constructor TMathImage.Create(AOwner:TComponent);
var num:integer;  bitmap:tbitmap;
begin
  inherited create(AOWner);
  bitmap:=tbitmap.create;
  bitmap.width:=width; bitmap.height:=height;
  picture.graphic:=bitmap;
  bitmap.free;
  Surface:=nil;
  SurfaceAllocated:=false;
  hregion:=0;
  x1d2:=-1; x2d2:=1; y1d2:=-1; y2d2:=1;
  x1d3:=-1; x2d3:=1; y1d3:=-1; y2d3:=1;
  z1d3:=-1; z2d3:=1; axisglb:=false; alpha:=6;
  vd:=6.4; zrd3:=45; yrd3:=45;
  fversion:='1.1  Oct 96';
end;



Destructor TMathImage.Destroy;
var i,j:integer;
begin
  if surface<>nil then
  begin
    for i:=0 to 200 do
    if surface^[i]<>nil then
    begin
      for j:=0 to 200 do
      begin
        if surface^[i]^[j]<>nil
        then
        begin
          dispose(surface^[i]^[j]);
          surface^[i]^[j]:=nil;
        end;
      end;
      dispose(surface^[i]);
      surface^[i]:=nil;
    end;
    dispose(surface);
  end;
  deleteobject(hregion);
  inherited destroy;
end;

procedure TMathImage.setaxisglb;
var r:Trect;
begin
  axisglb:=a;
  resetworld;
  if axisglb then
  begin
    deleteobject(hregion);
    hregion:=CreateRectRgn(windowx(x1d2),windowy(y2d2),windowx(x2d2),windowy(y1d2));
    SelectClipRgn(canvas.handle,hregion);
  end else
  begin
    deleteobject(hregion);
    hregion:=0;
    SelectClipRgn(canvas.handle,0);
  end;
end;

procedure TMathImage.setworld;
var w:integer;
 procedure xerror;
 begin
   if not (csDesigning in ComponentState) then
     application.messagebox('Error: d2worldx1>=d2worldx2','MathImage Error',mb_OK);
     x2d2:=x1d2+1;
 end;

 procedure yerror;
 begin
    if not (csDesigning in ComponentState) then
     application.messagebox('Error: d2worldy1>=d2worldy2','MathImage Error',MB_OK);
     y2d2:=y1d2+1;
 end;
begin
  x1d2:=x1; x2d2:=x2; y1d2:=y1; y2d2:=y2;
  canvas.font.size:=canvas.font.size-1; {There's one pointer in a VCL here that doesn't get freed}
  maxtw:=canvas.textwidth(floattostrf(y1d2,ffgeneral,3,3)); {see above}
  w:=canvas.textwidth(floattostrf(y1d2+(y2d2-y1d2)/4,ffgeneral,3,3));
  if w>maxtw then maxtw:=w;
  w:=canvas.textwidth(floattostrf(y1d2+(y2d2-y1d2)/2,ffgeneral,3,3));
  if w>maxtw then maxtw:=w;
  w:=canvas.textwidth(floattostrf(y1d2+3*(y2d2-y1d2)/4,ffgeneral,3,3));
  if w>maxtw then maxtw:=w;
  w:=canvas.textwidth(floattostrf(y2d2,ffgeneral,3,3));
  if w>maxtw then maxtw:=w;
  maxth:=canvas.textheight('-1.11');
  canvas.font.size:=canvas.font.size+1;
  If x2d2<=x1d2 then xerror;
  if axisglb then
    Bx := (width-7-maxtw-maxtw div 2)/(X2d2 - X1d2)
  else
    Bx:=width/(x2d2-x1d2);
  if axisglb then
  Ax :=6+maxtw-X1d2*Bx else
  Ax :=  - X1d2 * Bx;
  If y2d2<=y1d2 then yerror;
  if axisglb then
  By:=(height-7-maxth-maxth div 2)/(y1d2-Y2d2) else
  By :=height / (Y1d2 - Y2d2);
  if axisglb then
  Ay:=maxth div 2 +1 -By*Y2d2 else
  Ay := - Y2d2 * By;
end;

procedure TMathImage.resetworld;
begin
  setworld(x1d2,y1d2,x2d2,y2d2);
end;

procedure TMathImage.reset;
begin
  resetworld;
  d3resetworld;
  picture.bitmap.height:=height;
  picture.bitmap.width:=width;
end;

procedure TMathImage.setcolor;
begin
  canvas.pen.color:=color;
end;

function TMathImage.getcolor;
begin
  result:=canvas.pen.color;
end;

function TMathImage.WindowX;
var Temp: extended;
begin
  Temp := Ax + Bx * X;
  if abs(temp)<3000 then
    result:= round(Temp)
  else
    if temp<0 then result:=-3000 else result:=3000;
end;

function TMathImage.WindowY;
var Temp : extended;
begin
  Temp := Ay + By * Y;
  if abs(temp)<3000 then
    result:= round(Temp)
  else
    if temp<0 then result:=-3000 else result:=3000;
end;

function TMathImage.norm;
begin
  result:=sqrt(sqr(x)+sqr(y));
end;

function TMathImage.worldx;
begin
  result:=(xs-Ax)/Bx;
end;

function TMathImage.worldy;
begin
  result:=(ys-ay)/by;
end;

procedure TMathImage.clear;
var bitmap:tbitmap;
begin
  bitmap:=tbitmap.create;
  bitmap.width:=width; bitmap.height:=height;
  picture.graphic:=bitmap;
  bitmap.free; {This line should not be here according to
                the Delphi documentation, but without it
                the component leaks memory.}
  {canvas.fillrect(rect(0,0,width,height));}
  {The commented line above also clears the picture
   but leads to another memory leak}
end;

procedure TMathImage.DrawPoint;
begin
   SelectClipRgn(canvas.handle,hregion);
  canvas.pixels[windowx(x),windowy(y)]:=canvas.pen.color;
end;

procedure TMathImage.MovetoPoint;
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.moveto(windowx(x),windowy(y));
end;

procedure TMathImage.DrawLine;
var xw,yw:integer;
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.moveto(windowx(x1),windowy(y1));
  xw:=windowx(x2); yw:=windowy(y2);
  canvas.lineto(xw,yw);
  canvas.pixels[xw,yw]:=canvas.pen.color;
end;


procedure TMathImage.DrawLineto(x,y:extended);
begin
  SelectClipRgn(canvas.handle,hregion);
  canvas.Lineto(windowx(x),windowy(y));
end;

procedure TMathImage.drawaxes;
var xs,ys:integer; i,istart,ticks,savecolor:longint;
    t:string; itemp,xtick,ytick:extended;
begin
  savecolor:=canvas.pen.color;
  setaxisglb(true);
  deleteobject(hregion);
  hregion:=0;
  selectclipRgn(canvas.handle,0);
  canvas.pen.color:=axescolor;
  drawline(x1d2,y1d2,x2d2,y1d2);
  itemp:=ln((d2worldx2-d2worldx1)/8)/ln(10);
  if itemp>=0 then
   i:=trunc(itemp) else i:=trunc(itemp)-1;
  xtick:=exp(i*ln(10));
  itemp:=ln((d2worldy2-d2worldy1)/8)/ln(10);
  if itemp>=0 then
    i:=trunc(itemp) else i:=trunc(itemp)-1;
  ytick:=exp(i*ln(10));
  if xtick>0 then
  begin
    istart:=round(x1d2/xtick);
    i:=istart;
    ticks:=round((x2d2-x1d2)/xtick);
    with canvas.font do
     size:=size-1;
    if ticks<=2000 then
    repeat
      xs:=windowx(i*xtick);
      ys:=windowy(y1d2);
      canvas.moveto(xs,ys);
      canvas.lineto(xs,ys+4);
      if (i-istart) mod (ticks div 4) =0 then
      begin
        t:=floattostrf(i*xtick,ffgeneral,3,3);
        with canvas do
        begin
          textout(xs-(textwidth(t) div 2), ys+6,t);
          moveto(xs,ys);
          lineto(xs,ys+6);
        end;
      end;
      inc(i)
    until i*xtick>x2d2;
  end;
  with canvas.font do size:=size+1;
  xs:=windowx(x2d2);
  ys:=windowy(y1d2);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys-6);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys+6);
  canvas.textout(xs-canvas.textwidth(xlabel)-4,ys-canvas.textheight(xlabel)-6,xlabel);
  drawline(x1d2,y1d2,x1d2,y2d2);
  if ytick>0 then
  begin
    istart:=round(y1d2/ytick);
    i:=istart;
    ticks:=round((y2d2-y1d2)/ytick);
    with canvas.font do
      size:=size-1;
    if ticks <=2000 then
    repeat
      xs:=windowx(x1d2);
      ys:=windowy(i*ytick);
      canvas.moveto(xs,ys);
      canvas.lineto(xs-4,ys);
      if (i-istart) mod (ticks div 4) =0 then
      begin
        t:=floattostrf(i*ytick,ffgeneral,3,3);
        with canvas do
        begin
          textout(xs-textwidth(t)-6,ys-textheight(t) div 2,t);
          moveto(xs,ys);
          lineto(xs-6,ys);
        end;
      end;
      inc(i);
    until i*ytick>y2d2;
  end;
  with canvas.font do
    size:=size+1;
  xs:=windowx(x1d2);
  ys:=windowy(y2d2);
  canvas.moveto(xs,ys);
  canvas.lineto(xs+6,ys+6);
  canvas.moveto(xs,ys);
  canvas.lineto(xs-6,ys+6);
  canvas.textout(xs+8,ys,ylabel);
  if zerolines then
  begin
    canvas.pen.color:=zerolinescolor;
    drawline(0,y1d2,0,y2d2);
    drawline(x1d2,0,x2d2,0);
  end;
  canvas.pen.color:=savecolor;
  setaxisglb(true);
end;

procedure TMathImage.drawvector;
var aw,bw,xw,yw,u1,u2,v1,v2:integer; n:extended;
begin
  SelectClipRgn(canvas.handle,hregion);
  v1:=windowx(a+x);
  v2:=windowy(b+y);
  xw:=windowx(x);
  yw:=windowy(y);
  canvas.moveto(xw,yw);
  canvas.lineto(v1,v2);
  aw:=v1-xw;
  bw:=v2-yw;
  n:=norm(bw-aw,aw+bw);
  u1:=round(8.0*(bw-aw)/n);
  u2:=round(8.0*(-bw-aw)/n);
  canvas.moveto(v1,v2);
  canvas.lineto(v1+u1,v2+u2);
  u1:=round(8.0*(-aw-bw)/n);
  u2:=round(8.0*(aw-bw)/n);
  canvas.moveto(v1,v2);
  canvas.lineto(v1+u1,v2+u2);
end;


function max(x,y:extended):extended;
begin
  if x<y then result:=y else result:=x;
end;

function min(x,y:extended):extended;
begin
  if x<y then result:=x else result:=y;
end;


procedure TMathImage.makeradians;
  procedure d3worlderror;
  begin
    if x1d3>=x2d3 then
    begin
      application.messagebox('Error: d3worldx1>=d3worldx2','MathImage Error',mb_OK);
      x2d3:=x1d3+1;
    end;
    if y1d3>=y2d3 then
    begin
      application.messagebox('Error: d3worldy1>=d3worldy2','MathImage Error',mb_OK);
      y2d3:=y1d3+1;
    end;
    if z1d3>=z2d3 then
    begin
      application.messagebox('Error: d3worldz1>=d3worldz2','MathImage Error',mb_OK);
      z2d3:=z1d3+1;
    end;
    ad3:=max(max(x2d3-x1d3,y2d3-y1d3),z2d3-z1d3);
  end;

begin
  thetaz:=2*pi*zrd3/360;
  thetay:=2*pi*yrd3/360;
  arad:=pi*alpha/360;
  sinz:=sin(thetaz); cosz:=cos(thetaz);
  siny:=sin(thetay); cosy:= cos(thetay);
  tana:=sin(arad)/cos(arad);
  rightz:=(zrd3+90) - 180*trunc((zrd3+90.0)/180);
  righty:=yrd3 - 180*trunc(yrd3/180);
  ad3:=max(max(x2d3-x1d3,y2d3-y1d3),z2d3-z1d3);
  if ad3<=0 then d3worlderror;
  ad3:=2/ad3;
  bxd3:=-ad3*(x1d3+x2d3)/2;
  byd3:=-ad3*(y1d3+y2d3)/2;
  bzd3:=-ad3*(z1d3+z2d3)/2;
  ap:=min(height,width)/2/tana/vd;
  bxp:=width/2; byp:=height/2;
end;

function TMathImage.scalar(xb,yb,zb:extended):extended;
begin
  scalar:=yb*sinz*siny+zb*cosy+xb*siny*cosz;
end;

function TMathImage.dist(xb,yb,zb:extended):extended;
begin
  dist:=d3viewdist-scalar(xb,yb,zb);
end;

function TMathImage.d3distancetoviewer(x,y,z:extended):extended;
var xb,yb,zb:extended;
begin
  blockx(x,xb); blocky(y,yb); blockz(z,zb);
  d3distancetoviewer:=sqrt(sqr(d3viewdist*siny*sinz-yb)+
    sqr(d3viewdist*cosy-zb)+sqr(d3viewdist*siny*cosz-xb));
end;


procedure TMathImage.findbase(var i1,i2,i3:integer);
var dmax,d:extended; i,j,k:integer;
begin
  i1:=-1;i2:=-1;i3:=-1;
  dmax:=0;
  for i:=0 to 1 do
  for j:=0 to 1 do
  for k:=0 to 1 do
  begin
    d:=dist(-1+2*i,-1+2*j,-1+2*k);
    dmax:=max(dmax,d);
    if d=dmax then
    begin
      i1:=-1+2*i;i2:=-1+2*j;i3:=-1+2*k;
    end;
  end;
end;

procedure TMathImage.initworld;
var umin,umax,vmin,vmax,d2w:extended;
    i1,i2,i3:integer;
    i,j,k:integer;
begin
  if d3viewdist<0 then d3viewdist:=0.0000001;
  if alpha > 179 then alpha:=179;
  if alpha <0.1 then alpha:=0.1;
  makeradians;
  findbase(i1,i2,i3);
    if i1=-1 then basex:=x1d3 else basex:=x2d3;
    if i2=-1 then basey:=y1d3 else basey:=y2d3;
    if i3=-1 then basez:=z1d3 else basez:=z2d3;
    if i1=1 then frontx:=x1d3 else frontx:=x2d3;
    if i2=1 then fronty:=y1d3 else fronty:=y2d3;
    if i3=1 then frontz:=z1d3 else frontz:=z2d3;
end;

procedure TMathImage.d3setworld;
var d:extended;
begin
  x1d3:=x1;
  x2d3:=x2;
  y2d3:=y2;
  y1d3:=y1;
  y2d3:=y2;
  z1d3:=z1;
  z2d3:=z2;
  zrd3:=zrot; yrd3:=yrot; vd:=vdist; alpha:=vangle;
  initworld;
end;

procedure TMathImage.d3resetworld;
begin
  d3setworld(x1d3,y1d3,z1d3,x2d3,y2d3,z2d3,
     vd,alpha,zrd3,yrd3);
end;

procedure TMathImage.blockx(x:extended;var xb:extended);
begin
  xb:=bxd3+ad3*x;
end;

procedure TMathImage.blocky(y:extended;var yb:extended);
begin
  yb:=byd3+ad3*y;
end;

procedure TMathImage.blockz(z:extended;var zb:extended);
begin
  zb:=bzd3+ad3*z;
end;

procedure TMathImage.d3window(x,y,z:extended; var xs,ys:integer);
var xb,yb,zb,rad,tan,tempx,tempy,d,u,v:extended;
begin
  blockx(x,xb);
  blocky(y,yb);
  blockz(z,zb);
  project(xb,yb,zb,u,v);
  tempx:=bxp+ap*u;
  if abs(tempx)<3000 then xs:=round(tempx)
  else if tempx<0 then xs:=-3000 else xs:=3000;
  tempy:=byp-ap*v;
  if abs(tempy)<3000 then ys:=round(tempy)
  else if tempy <0 then ys:=-3000 else ys:=3000;
end;

procedure TMathImage.project;
var scal,d:extended;
begin
  scal:=scalar(xb,yb,zb);
  d:=d3viewdist-scal;
  if righty<>0 then
      v:=(zb-scal*cosy)/siny
  else
      v:=-(yb*sinz+xb*cosz)/cosy;
  if rightz<>0 then
  u:=(Yb+sinz*(v*cosy-scal*siny))/cosz
    else
      u:=-Xb*sinz;
  if d<=0 then d:=1.e-10;
  u:=u/d;
  v:=v/d;
end;

procedure TMathImage.d3moveto(x,y,z:extended);
var xs,ys:integer; visible:boolean;
begin
  d3window(x,y,z,xs,ys);
  canvas.moveto(xs,ys);
end;

procedure TMathImage.d3drawpoint(x,y,z:extended);
var xs,ys:integer;
begin
  d3window(x,y,z,xs,ys);
  canvas.pixels[xs,ys]:=canvas.pen.color;
end;

procedure TMathImage.d3drawline(x1,y1,z1,x2,y2,z2:extended);
var u1,v1,u2,v2:integer;
begin
  d3window(x1,y1,z1,u1,v1);
  d3window(x2,y2,z2,u2,v2);
  canvas.moveto(u1,v1);
  canvas.lineto(u2,v2);
  canvas.pixels[u2,v2]:=canvas.pen.color;
end;

procedure TMathImage.d3drawlineto(x,y,z:extended);
var xs,ys:integer;
begin
    d3window(x,y,z,xs,ys);
    canvas.lineto(xs,ys);
end;


procedure TMathImage.drawoneaxis(x1,y1,z1,x2,y2,z2:extended;c:string);
var norms,wx,wy:extended;
    xs1,ys1,xs2,ys2:integer; vsx,vsy:extended;
begin
  d3drawline(x1,y1,z1,x2,y2,z2);
  d3window(x1,y1,z1,xs1,ys1);
  d3window(x2,y2,z2,xs2,ys2);
  vsx:=(xs2-xs1); vsy:=(ys2-ys1);
  norms:=sqrt(vsx*vsx+vsy*vsy);
  if norms>0 then
  begin
    vsx:=vsx/norms; vsy:=vsy/norms;
    wx:=(-vsx+vsy)/sqrt(2); wy:=(-vsy-vsx)/sqrt(2);
    canvas.moveto(xs2,ys2);
    canvas.lineto(xs2+round(5*wx),ys2+round(5*wy));
    wx:=(-vsx-vsy)/sqrt(2); wy:=(-vsy+vsx)/sqrt(2);
    canvas.moveto(xs2,ys2);
    canvas.lineto(xs2+round(5*wx),ys2+round(5*wy));
    canvas.textout(xs2-10,ys2-10,c);
  end;
end;


procedure TMathImage.d3drawaxes(c1,c2,c3:string);


begin   {******* drawd3axes ******}
    drawoneaxis(x1d3,y1d3,z1d3,x2d3,y1d3,z1d3,c1);
    drawoneaxis(x1d3,y1d3,z1d3,x1d3,y2d3,z1d3,c2);
    drawoneaxis(x1d3,y1d3,z1d3,x1d3,y1d3,z2d3,c3);
end;

procedure TMathImage.d3drawzerocross;
begin
   if 0>=x1d3 then if 0<=x2d3 then if 0>=z1d3 then if 0<=z2d3 then
   d3drawline(0,y1d3,0,0,y2d3,0);
   if 0>=z1d3 then if 0<=z2d3 then if 0>=y1d3 then if 0<=y2d3 then
   d3drawline(x1d3,0,0,x2d3,0,0);
   if 0>=y1d3 then if 0<=y2d3 then if 0>=x1d3 then if 0<=x2d3 then
   d3drawline(0,0,z1d3,0,0,z2d3);
end;

procedure TMathImage.d3drawworldbox;
var i:integer; delta:extended; savestyle:tpenstyle;
savecolor:tcolor;
begin
  savestyle:=canvas.pen.style;
  canvas.pen.style:=pssolid;
  savecolor:=canvas.pen.color;
  canvas.pen.color:=clblack;
    d3drawline(basex,basey,basez,frontx,basey,basez);
    d3drawline(basex,basey,basez,basex,fronty,basez);
    d3drawline(basex,basey,basez,basex,basey,frontz);
    d3drawline(basex,fronty,basez,frontx,fronty,basez);
    d3drawline(basex,fronty,basez,basex,fronty,frontz);
    d3drawline(basex,basey,frontz,frontx,basey,frontz);
    d3drawline(basex,basey,frontz,basex,fronty,frontz);
    d3drawline(frontx,basey,basez,frontx,fronty,basez);
    d3drawline(frontx,basey,basez,frontx,basey,frontz);
 canvas.pen.style:=savestyle;
 canvas.pen.color:=savecolor;
end;

procedure TMathImage.d3createSurfaceMem(xmesh,ymesh:integer; var created:boolean);
var i,j:integer; xx,yy,zz:extended;
begin
  if (xmesh>200) or (xmesh<1) or (ymesh<1) or (ymesh>200) then
  created:=false else
  begin
    if surface<>nil then
    begin
      for i:=0 to 200 do
      begin
        if surface^[i]<>nil then
        begin
          for j:=0 to 200 do
          begin
            if surface^[i]^[j]<>nil then
              dispose(surface^[i]^[j]);
            surface^[i]^[j]:=nil;
          end;
          dispose(surface^[i]);
        end;
        surface^[i]:=nil;
      end;
      dispose(surface);
    end;
    surface:=nil;
    surface:=new(PSurface);
    for i:=0 to 200 do
      surface^[i]:=nil;
    for i:=0 to xmesh do
    begin
      surface^[i]:=new(PPointarray);
      for j:=0 to 200 do
        surface^[i]^[j]:=nil;
      for j:=0 to ymesh do
        surface^[i]^[j]:=new(PD3Point);
    end;
    xm:=xmesh; ym:=ymesh;
    created:=true; surfaceallocated:=true;
    for i:=0 to xmesh do
    for j:=0 to ymesh do
    with surface^[i]^[j]^ do
    begin
      xx:=x1d3+i*(x2d3-x1d3)/xmesh;
      yy:=y1d3+j*(y2d3-y1d3)/ymesh;
      zz:=x1d3+i*(x2d3-x1d3)/xmesh;
      d3window(xx,yy,zz,xw,yw);
      dist:=d3distancetoviewer(xx,yy,zz);
      doshow:=true;
    end;
  end;
end;

procedure TMathImage.d3makesurfacepoint;
begin
  if surface=nil then made:=false
  else
  begin
    if surface^[i]=nil then made:=false
    else
    begin
      if surface^[i]^[j]=nil then made:=false else
      with surface^[i]^[j]^ do
      begin
        d3window(x,y,z,xw,yw);
        dist:=d3distancetoviewer(x,y,z);
        made:=true;
      end;
    end;
  end;
end;

function TMathImage.inspanned(p1,p2,p3,q:TD3Point):boolean;
  {decides whether the point q
   is hidden by the triangle p1,p2,p3}
var u1,u2,v1,v2,x1,x2,denom,c1,c2:extended;

begin
  if q.dist<p1.dist then begin result:=false; exit; end;
  u1:=p3.xw-p1.xw; v1:=p2.xw-p1.xw;
  u2:=p3.yw-p1.yw; v2:=p2.yw-p1.yw;
  x1:=q.xw-p1.xw;
  x2:=q.yw-p1.yw;
  denom:=u1*v2-v1*u2;
  if denom=0 then
  begin result:=false; exit; end
  else
  begin
    c2:=(u1*x2-u2*x1)/denom;
    if c2<0 then begin result:=false; exit; end;
    c1:=(v2*x1-v1*x2)/denom;
    if c1<0 then begin result:=false; exit; end;
    result:=(c1+c2<=1);
  end;
end; {InSpanned}

procedure TMathImage.d3MakeHideInfo;
   var i,j,k,l,nmin,nmax,n,m:integer;
       p1,p2,p3,q:TD3Point;
       show:boolean;
begin
  if surfaceallocated
  then
  begin
    for i:=0 to xm do
    for j:=0 to ym do
    surface^[i]^[j]^.doshow:=true;
    for i:=0 to xm do
    for j:=0 to ym do
    begin
      if not deaf then application.processmessages;
      q:=surface^[i]^[j]^;
      for k:=0 to xm-1 do
      for l:=0 to ym-1 do
      begin
        p1:=surface^[k]^[l]^;
        p2:=surface^[k+1]^[l]^;
        p3:=surface^[k]^[l+1]^;
        if p1.doshow or p2.doshow or p3.doshow then
        if ((p1.xw<>q.xw) or (p1.yw<>q.yw) or (p1.dist<>q.dist))
        and ((p2.xw<>q.xw) or (p2.yw<>q.yw) or (p2.dist<>q.dist))
        and ((p3.xw<>q.xw) or (p3.yw<>q.yw) or (p3.dist<>q.dist))
        then
        begin
          if inspanned(p1,p2,p3,q) then
          begin
            surface^[i]^[j]^.doshow:=false;
            k:=xm-1; l:=ym-1;
          end else
          begin
            p1:=surface^[k+1]^[l+1]^;
            if p1.doshow or p2.doshow or p3.doshow then
            if ((p1.xw<>q.xw) or (p1.yw<>q.yw) or (p1.dist<>q.dist))
            and ((p2.xw<>q.xw) or (p2.yw<>q.yw) or (p2.dist<>q.dist))
            and ((p3.xw<>q.xw) or (p3.yw<>q.yw) or (p3.dist<>q.dist))
            then
            begin
              if inspanned(p1,p2,p3,q) then
              begin
                surface^[i]^[j]^.doshow:=false;
                k:=xm-1; l:=ym-1;
              end;
            end;
          end;
        end;
      end;
    end;
  end;
end;

function TMathImage.d3visible;
begin
  result:=false;
  if surface<>nil then if
  surface^[i]<>nil then if
  surface^[i]^[j]<>nil then
  if surface^[i]^[j]^.doshow
  then result:=true;
end;

procedure TMathImage.d3drawsurface(fill,deaf:boolean);
var xs,ys,xrs,yrs,xfs,yfs,i,j,n,nmin,nmax:integer;
    poly:array[1..4] of TPoint;
    color,savecolor:longint;
begin
  savecolor:=canvas.pen.color;
  if fill then
  if surfaceallocated then
  for i:=0 to xm-1 do
  begin
    repaint;
    if not deaf then application.processmessages;
    for j:=0 to ym-1 do
    begin
      xs:=surface^[i]^[j]^.xw; ys:=surface^[i]^[j]^.yw;
      xrs:=surface^[i]^[j+1]^.xw; yrs:=surface^[i]^[j+1]^.yw;
      xfs:=surface^[i+1]^[j]^.xw; yfs:=surface^[i+1]^[j]^.yw;
      poly[1]:=point(xs,ys); poly[2]:=point(xrs,yrs);
      xs:=surface^[i+1]^[j+1]^.xw; ys:=surface^[i+1]^[j+1]^.yw;
      poly[3]:=point(xs,ys);
      poly[4]:=point(xfs,yfs);
      color:=canvas.brush.color;
      canvas.pen.color:=color;
      canvas.polygon(poly);
    end; {for j}
  end;
  if surfaceallocated then
  for i:=0 to xm do
  begin
    if not deaf then
    begin
      repaint;
      application.processmessages;
    end;
    for j:=0 to ym do
    begin
      canvas.pen.color:=savecolor;
      if fill then
      begin
        if surface^[i]^[j]^.doshow then
        begin
          if  j<ym then if surface^[i]^[j+1]^.doshow then
          begin
            canvas.moveto(surface^[i]^[j]^.xw,surface^[i]^[j]^.yw);
            canvas.lineto(surface^[i]^[j+1]^.xw,surface^[i]^[j+1]^.yw);
          end;
          if i<xm then if surface^[i+1]^[j]^.doshow then
          begin
            canvas.moveto(surface^[i]^[j]^.xw,surface^[i]^[j]^.yw);
            canvas.lineto(surface^[i+1]^[j]^.xw,surface^[i+1]^[j]^.yw);
          end;
        end;
      end {if fill}
      else
      begin
        canvas.moveto(surface^[i]^[j]^.xw,surface^[i]^[j]^.yw);
        if j<ym then canvas.lineto(surface^[i]^[j+1]^.xw,surface^[i]^[j+1]^.yw);
        canvas.moveto(surface^[i]^[j]^.xw,surface^[i]^[j]^.yw);
        if i<xm then canvas.lineto(surface^[i+1]^[j]^.xw,surface^[i+1]^[j]^.yw);
      end; {if not fill}
    end; {for j}
  end; {for n,i}
  canvas.pen.color:=savecolor;
end;

end.
